/**
* \file: configuration.c
*
* \version: $Id:$
*
* \release: $Name:$
*
* \component: automounter
*
* \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
*
* \copyright (c) 2010, 2011 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/
#include <sys/mount.h>
#include <string.h>
#include <argp.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <unistd.h>
#include <grp.h>

#include "control/automounter.h"
#include "control/configuration.h"
#include "control/configuration_defaults.h"
#include "control/blacklist.h"
#include "utils/version_info.h"
#include "utils/helper.h"

typedef struct mount_options_entry_t mount_options_entry_t;

typedef struct mount_options_entry_t
{
	char *file_system;
	mount_options_entry_t *next_entry;
	char options[];
} mount_options_entry_t;

typedef struct configuration_data_t
{
	char *conf_filename;

	const char *proc_name;

	logger_loglevel_t log_level;

	bool log_level_already_set;

	bool console_enabled;

	bool shall_daemonize;

	bool mount_mixed_cdda;

	char *media_dir;

	char *automount_options;

	mount_options_entry_t *fs_specific_options_list;

	__mode_t socket_dir_mode;

	__mode_t socket_mode;

	gid_t socket_group_id;

} configuration_data_t;

/**
 * instance of the data structure filled with default values. NULL is used in case of character array to signal the module to select the default value.
 */
static configuration_data_t data =
{
		NULL, 							//conffile
		NULL, 							//proc_name
		DEFAULT_LOG_LEVEL,				//log_level
		false,							//log_level_already_set
		DEFAULT_CONSOLE_ENABLED,		//console_enabled
		DEFAULT_SHALL_DAEMONIZE,		//shall_daemonize
		DEFAULT_MOUNT_MIXED_MODE_CDS,	//mount_mixed_cdda
		NULL,							//media_dir
		NULL,							//automount_options
		NULL,							//fs_specific_options_list
		//octal representation -> leading 0
		0700,							//socket_dir_mode
		0600,							//socket_mode
		0								//socket_group_id
};

static const char * const configuration_doc =
		"ADIT automounter daemon - A daemon for automatically mounting hot plug mass storage devices.";

/*
 * The options we understand.
 */
static const struct argp_option const configuration_sync_options[] =
{
		{"loglevel",  'l', "<loglevel>" ,      0,  "Start the automounter with a given log level. Supported values are: none, error, info, debug",0},
		{"quiet",  'q', NULL ,      0,  "Don't plot any status messages or log message to the console.",0},
		{"daemon",  'd', NULL ,      0,  "Start the automounter as daemon.",0},
		{"version",  'v', NULL ,      0,  "Prints out version information.",0},
		{"configuration", 'c', "<conf file", 0, "Load the configuration from given file. [default: /etc/automounter.conf",0},
		{NULL,0,NULL,0,NULL,0}
};

static error_code_t configuration_parse_args(int argc, char *argv[]);

static char *configuration_filter_help(int key, const char *text, void *input);

static error_t configuration_parse_opt (int key, char *arg, struct argp_state *state);

static error_code_t configuration_parse_log_level(const char *arg,bool override);

static void configuration_print_version(void);

static const char *configuration_check_for_default_file(void);

static void configuration_process_configuration_file_line(char *line_in_file, int line_no);

static error_code_t configuration_extract_kv_from_line(char *clean_line,char **key, char **value,int line_no);

static error_code_t configuration_extract_bool_from_string(const char *value_str, bool *result_ptr);

static error_code_t configuration_extract_group_id(const char *value, gid_t *group_id_ptr);

static error_code_t configuration_extract_mode(const char *value, __mode_t *mode_ptr);

static error_code_t configuration_extract_automount_options_per_fs(char *value);

static const char *configuration_find_mount_options_to_fs(const char *file_system);

static void configuration_add_options_per_fs(const char *fs, const char *options);

static void configuration_free_options_list(void);

//--------------------------------------- API members ------------------------------------
error_code_t configuration_init(int argc, char *argv[])
{
	data.proc_name=argv[0];
	return configuration_parse_args(argc,argv);
}

void configuration_deinit(void)
{
	if (data.media_dir!=NULL)
	{
		free(data.media_dir);
		data.media_dir=NULL;
	}

	if (data.automount_options!=NULL)
	{
		free(data.automount_options);
		data.automount_options=NULL;
	}

	configuration_free_options_list();

	if (data.conf_filename!=NULL)
	{
		free(data.conf_filename);
		data.conf_filename=NULL;
	}
}

error_code_t configuration_parse_configuration_file(void)
{
	int line_no=1;
	error_code_t result=RESULT_OK;
	const char *conf_file;
	FILE* conf_file_fd;
	char *line_in_file=NULL;
	size_t line_buf_size;

	conf_file=data.conf_filename;

	//no conf file passed as command line option -> check for default
	if (conf_file==NULL)
		conf_file=configuration_check_for_default_file();

	//no file defined and none found at the default place -> nothing to parse
	if (conf_file==NULL)
		return RESULT_OK;

	conf_file_fd=fopen(conf_file,"r");
	if (conf_file_fd==NULL)
	{
		logger_log_error("Unable to read the configuration file: %s.",conf_file);
		return RESULT_INVALID_ARGS;
	}

	while (getline(&line_in_file,&line_buf_size,conf_file_fd)!=-1)
	{
		configuration_process_configuration_file_line(line_in_file,line_no);
		line_no++;
	}

	if (line_in_file!=NULL)
		free(line_in_file);

	fclose(conf_file_fd);

	return result;
}

const char *configuration_get_proc_name(void)
{
	return data.proc_name;
}

const char *configuration_get_automount_options(const char* file_system)
{
	const char *automount_options=NULL;

	//try to find automount options matching to given file system
	if (file_system!=NULL)
		automount_options=configuration_find_mount_options_to_fs(file_system);

	//if nothing was found this way, take the generic automounter options
	if (automount_options==NULL)
		automount_options=data.automount_options;

	//if the generic automounter options are not set, take the implementation default settings
	if (automount_options!=NULL)
		return automount_options;
	else
		return DEFAULT_AUTOMOUNT_OPTIONS;
}

const char *configuration_get_media_dir(void)
{
	if (data.media_dir!=NULL)
		return data.media_dir;
	else
		return DEFAULT_MEDIA_DIR;
}

bool configuration_mount_data_cdda_cds(void)
{
	return data.mount_mixed_cdda;
}

bool configuration_shall_daemonize(void)
{
	return data.shall_daemonize;
}

bool configuration_is_console_enabled(void)
{
	return data.console_enabled;
}

logger_loglevel_t configuration_get_loglevel(void)
{
	return data.log_level;
}

__mode_t configuration_get_mountpoint_mode(void)
{
	return DEFAULT_MOUNT_POINT_MODE;
}

int configuration_mp_ext_give_up_cnt(void)
{
	return DEFAULT_MP_EXT_GIVEUP_CNT;
}

__mode_t configuration_get_socket_dir_mode(void)
{
	return data.socket_dir_mode;
}

__mode_t configuration_get_socket_access_mode(void)
{
	return data.socket_mode;
}

gid_t configuration_get_socket_group_id(void)
{
	return data.socket_group_id;
}
//----------------------------------------------------------------------------------------

//--------------------------------------- internal members -------------------------------
static error_code_t configuration_parse_args(int argc, char *argv[])
{
	error_code_t result=RESULT_OK;
	struct argp argp =
	{
			configuration_sync_options,
			configuration_parse_opt,
			NULL,
			configuration_doc,
			NULL,
			configuration_filter_help,
			NULL
	};

	if (argp_parse(&argp, argc, argv, ARGP_NO_EXIT, 0, &result)==EINVAL)
		result=RESULT_INVALID_ARGS;

	return result;
}

static char *configuration_filter_help(int key, const char *text, void *input)
{
	error_code_t *result_ptr;
	result_ptr=(error_code_t *)input;

	if ((key & ARGP_KEY_HELP_POST_DOC)!=0)
		*result_ptr=RESULT_HELP_PRINTED;
	return (char *)text;
}

static error_t configuration_parse_opt (int key, char *arg, struct argp_state *state)
{
	error_code_t *result_ptr;
	result_ptr=(error_code_t *)state->input;

	switch (key)
	{
	case 'l':
		(*result_ptr)=configuration_parse_log_level(arg, true);
		if (*result_ptr==RESULT_INVALID_ARGS)
		{
			fprintf(stderr,"Invalid log level in command line.\n");
			argp_usage(state);
		}
		break;
	case 'q':
		data.console_enabled=false;
		break;
	case 'd':
		data.shall_daemonize=true;
		break;
	case 'v':
		configuration_print_version();
		(*result_ptr)=RESULT_HELP_PRINTED;
		break;
	case 'c':
		data.conf_filename=strdup(arg);
		if (data.conf_filename==NULL)
			(*result_ptr)=RESULT_NORESOURCE;
		break;
	default:
		return ARGP_ERR_UNKNOWN;
	}
	return 0;
}

static error_code_t configuration_parse_log_level(const char *value_str, bool override)
{
	error_code_t result;

	if (data.log_level_already_set && !override)
		return RESULT_OK;

	result=logger_parse_loglevel(value_str,&data.log_level);
	if (result==RESULT_OK)
		data.log_level_already_set=true;

	return result;
}

static void configuration_print_version(void)
{
	printf("%s\n\n",VERSION_INFO_FORMATED_LINE);
}

static const char *configuration_check_for_default_file(void)
{
	struct stat sb;
	if (stat(DEFAULT_CONF_FILE, &sb) == -1)
		//nothing found
		return NULL;
	return DEFAULT_CONF_FILE;
}

static void configuration_process_configuration_file_line(char *line_in_file, int line_no)
{
	char *key=NULL;
	char *value=NULL;

	char *clean_line;
	bool is_empty=true;

	//modifies the string in line_in_conf and returns a pointer to somewhere in the line
	//the result ensures that we have no leading spaces, no trailing spaces and no newline in the string
	clean_line=helper_trim(line_in_file, &is_empty);

	//do we have a comment here or is the line empty?
	if (clean_line[0]!='#' && !is_empty)
	{
		if (configuration_extract_kv_from_line(clean_line,&key,&value,line_no) == RESULT_OK)
		{
			if (strcasecmp(key,"BLACKLIST_DEVICE")==0)
				blacklist_add_blacklisted_device(value);
			else if (strcasecmp(key,"BLACKLIST_PARTITION")==0)
				blacklist_add_blacklisted_partition(value);
			else if (strcasecmp(key,"SUPPORTED_FILE_SYSTEMS")==0)
				blacklist_add_supported_fs_array(value);
			else if (strcasecmp(key,"MOUNT_MIXED_MODE_CDS")==0)
			{
				if (configuration_extract_bool_from_string(value, &data.mount_mixed_cdda)!=RESULT_OK)
					logger_log_error("Boolean value expected after MOUNT_MIXED_CDS= in configuration file "
							"(line %d). Ignoring the line!",line_no);
			}
			else if (strcasecmp(key,"LOGLEVEL")==0)
			{
				if (configuration_parse_log_level(value,false)!=RESULT_OK)
					logger_log_error("Invalid log level in configuration file (line %d): %s. Ignoring it!", line_no,value);
			}
			else if (strcasecmp(key,"MOUNT_POINT_DIR")==0)
			{
				if (data.media_dir!=NULL)
					free(data.media_dir);
				data.media_dir=strdup(value);
				if (data.media_dir==NULL)
					automounter_die_on_resource_issues();
			}
			else if (strcasecmp(key,"AUTOMOUNT_OPTIONS")==0)
			{
				if (data.automount_options!=NULL)
					free(data.automount_options);
				data.automount_options=strdup(value);
				if (data.automount_options==NULL)
					automounter_die_on_resource_issues();
			}
			else if (strcasecmp(key,"AUTOMOUNT_OPTIONS_PER_FS")==0)
			{
				if (configuration_extract_automount_options_per_fs(value)!=RESULT_OK)
				{
					logger_log_error("AUTOMOUNT_OPTIONS_PER_FS defined with invalid format in the "
							"configuration file (line %d).",line_no);
					logger_log_error("Expected format: AUTOMOUNT_OPTIONS_PER_FS=<options> [<file system>]",line_no);
					logger_log_error("Ignoring the line!");
				}
			}
			else if (strcasecmp(key,"IPC_SOCKET_GROUP")==0)
			{
				if (configuration_extract_group_id(value, &data.socket_group_id)!=RESULT_OK)
					logger_log_error("Unknown group or invalid group id defined in configuration "
							"file (line %d): %s. Ignoring it!", line_no,value);
			}
			else if (strcasecmp(key,"IPC_SOCKET_RIGHTS")==0)
			{
				if (configuration_extract_mode(value, &data.socket_mode)!=RESULT_OK)
					logger_log_error("Invalid socket access mode defined in configuration file "
							"(line %d): %s. Ignoring it!", line_no,value);
			}
			else if (strcasecmp(key,"IPC_SOCKET_DIR_RIGHTS")==0)
			{
				if (configuration_extract_mode(value, &data.socket_dir_mode)!=RESULT_OK)
					logger_log_error("Invalid socket directory access mode defined in configuration"
							" file (line %d): %s. Ignoring it!", line_no,value);
			}
			else
				logger_log_error("Unknown tag %s in configuration file found (line %d). Ignoring it!",key,line_no);
		}
	}
}

static error_code_t configuration_extract_kv_from_line(char *clean_line,char **key, char **value,int line_no)
{
	bool invalid_line_found;
	char *tmp;

	tmp=strchr(clean_line,'=');
	//no '=' found -> error
	invalid_line_found=(tmp==NULL);

	//split the line into two strings
	//would check invalid_line_found here but lin(t) ...
	if (tmp!=NULL)
	{
		*tmp='\0';
		//pointing now to value
		tmp++;
	}

	//trim key & check if empty
	if (!invalid_line_found)
		*key=helper_trim(clean_line, &invalid_line_found);

	//trim value (we allow empty value lines)
	if (!invalid_line_found)
		*value=helper_trim(tmp, NULL);

	if (invalid_line_found)	{
		logger_log_error("Syntax error in configuration file (line %d). (Expected format: <tag>=<value>)",line_no);
		logger_log_error("-> Ignoring invalid line starting with: %s",clean_line);
		return RESULT_INVALID_ARGS;
	}
	else
		return RESULT_OK;
}

static error_code_t configuration_extract_bool_from_string(const char *value_str, bool *result_ptr)
{
	if (strcasecmp(value_str,"true")==0)
		*result_ptr=true;
	else if (strcasecmp(value_str,"false")==0)
		*result_ptr=false;
	else if (*value_str=='1')
		*result_ptr=true;
	else if (*value_str=='0')
		*result_ptr=false;
	else
		return RESULT_INVALID_ARGS;

	return RESULT_OK;
}

static error_code_t configuration_extract_group_id(const char *value, gid_t *group_id_ptr)
{
	char *parse_result;
	struct group *group_lookup_result;

	//try first to parse for a number which is assumed to be the id
	*group_id_ptr=(gid_t)strtol(value, &parse_result, 10);

	if (parse_result!=value)
		return RESULT_OK;

	//given value is not a number, take the value as name and determine the group id
	group_lookup_result=getgrnam (value);

	if (group_lookup_result!=NULL)
	{
		*group_id_ptr=group_lookup_result->gr_gid;
		return RESULT_OK;
	}
	else
		return RESULT_INVALID_ARGS;
}

static error_code_t configuration_extract_mode(const char *value, __mode_t *mode_ptr)
{
	char *parse_result;
	*mode_ptr=(__mode_t)strtol(value, &parse_result, 8);
	if (parse_result==value)
		return RESULT_INVALID_ARGS;
	else
		return RESULT_OK;
}

static error_code_t configuration_extract_automount_options_per_fs(char *value)
{
	char *options;
	char *fs;
	char *tmp;

	options=value;
	fs=strstr(options, " [");
	if (fs==NULL)
		return RESULT_INVALID_ARGS;

	//terminate options
	*fs='\0';

	//move to begining of fs string
	fs+=2;

	//find ']' if there and replace it by \0 to terminate the fs string
	tmp=strrchr(fs, ']');
	if (tmp!=0)
		*tmp='\0';

	configuration_add_options_per_fs(fs, options);

	return RESULT_OK;
}

static void configuration_add_options_per_fs(const char *fs, const char *options)
{
	int lenfs;
	int lenoptions;
	mount_options_entry_t *new_entry;

	lenfs=strlen(fs);
	lenoptions=strlen(options);

	//len options + len fs + 2x'\0' + sizeof(mount_options_entry_t)
	new_entry=malloc(lenfs+lenoptions+2+sizeof(mount_options_entry_t));
	if (new_entry==NULL)
	{
		automounter_die_on_resource_issues();
		return;
	}

	//copy options to options item of structure
	strcpy(new_entry->options, options);

	//point fs after the end of options string
	new_entry->file_system=new_entry->options+lenoptions+1;

	//copy fs to item in structure
	strcpy(new_entry->file_system, fs);

	//link the entry into the list
	new_entry->next_entry=data.fs_specific_options_list;
	data.fs_specific_options_list=new_entry;
}

static const char *configuration_find_mount_options_to_fs(const char *file_system)
{
	mount_options_entry_t *entry=data.fs_specific_options_list;
	while (entry != NULL)
	{
		if (strcmp(entry->file_system, file_system)==0)
			return entry->options;
		entry=entry->next_entry;
	}

	return NULL;
}

static void configuration_free_options_list()
{
	mount_options_entry_t *entry=data.fs_specific_options_list;
	while (entry!=NULL)
	{
		data.fs_specific_options_list=entry->next_entry;
		free(entry);
		entry=data.fs_specific_options_list;
	}
}
//----------------------------------------------------------------------------------------
